﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.IO;
// Cal routine will prompt user with message box
using System.Windows.Forms;
using Agilent.AgM9391.Interop;
using Agilent.AgM938x.Interop;
using Ivi.Visa.Interop;


namespace NSM9391A
{
    public partial class Vsag_M9391A
    {
        public double getOutputAtten(double frequency, int path)
        {
            if (frequency <= outputCalFreqs[0])
                return outputCalValues[0, path];
            if (frequency >= outputCalFreqs[outputCalFreqs.Length - 1])
                return outputCalValues[outputCalFreqs.Length - 1, path];
            int index = 0;
            while (frequency > outputCalFreqs[index]) index++;
            return linear(frequency, outputCalFreqs[index - 1], outputCalFreqs[index], outputCalValues[index - 1, path], outputCalValues[index, path]);
        }

        public double getInputAtten(double frequency, int path)
        {
            if (frequency <= inputCalFreqs[0])
                return inputCalValues[0, path];
            if (frequency >= inputCalFreqs[inputCalFreqs.Length - 1])
                return inputCalValues[inputCalFreqs.Length - 1, path];
            int index = 0;
            while (frequency > inputCalFreqs[index]) index++;
            return linear(frequency, inputCalFreqs[index - 1], inputCalFreqs[index], inputCalValues[index - 1, path], inputCalValues[index, path]);
        }

        // Linaer interpoation
        public double linear(double x, double x0, double x1, double y0, double y1)
        {
            if ((x1 - x0) == 0)
            {
                return (y0 + y1) / 2;
            }
            return y0 + (x - x0) * (y1 - y0) / (x1 - x0);
        }

        // Load cal data with some default values
        public void initCalData(double inputLoss = 0, double outputLoss = 0)
        {
            int numInputCalPoints = 6;
            int numOutputCalPoints = 6;
            int numInputCalPaths = 1;
            int numOutputCalPaths = 1;

            // Clean up any old cal data
            if (inputCalFreqs != null) inputCalFreqs = null;
            if (inputCalValues != null) inputCalValues = null;
            if (outputCalFreqs != null) outputCalFreqs = null;
            if (outputCalValues != null) outputCalValues = null;

            // Size the Frequency arrays
            inputCalFreqs = new double[numInputCalPoints];
            inputCalValues = new double[numInputCalPoints, numInputCalPaths];
            outputCalFreqs = new double[numOutputCalPoints];
            outputCalValues = new double[numOutputCalPoints, numOutputCalPaths];

            // Set the input loss data
            for (int i = 0; i < numInputCalPoints; i++)
            {
                inputCalFreqs[i] = 1e9 * (i + 1);
                for (int j = 0; j < numInputCalPaths; j++)
                {
                    inputCalValues[i, j] = inputLoss;
                }
            }

            // Set the output loss data
            for (int i = 0; i < numOutputCalPoints; i++)
            {
                outputCalFreqs[i] = 1e9 * (i + 1);
                for (int j = 0; j < numOutputCalPaths; j++)
                {
                    outputCalValues[i, j] = outputLoss;
                }
            }
        }

        // readCalData
        // Reads a calibration data file with the following format:
        /*
                Input Freqs,6,
                Input Paths,2,
                Output Freqs,6,
                Output Paths,2,
                ,,
                Input Data,,
                Frequency,Path 1,path 2
                1.00E+09,0.11,0.21
                2.00E+09,0.12,0.22
                3.00E+09,0.13,0.23
                4.00E+09,0.14,0.24
                5.00E+09,0.15,0.25
                6.00E+09,0.16,0.26
                ,,
                Output Data,,
                Frequency,path 1,Path 2
                1.00E+09,6.11,6.21
                2.00E+09,6.12,6.22
                3.00E+09,6.13,6.23
                4.00E+09,6.14,6.24
                5.00E+09,6.15,6.25
                6.00E+09,6.16,6.26
        */
        public void readCalData(string filePath, string fileName)
        {
            int numInputCalPoints = 0;
            int numOutputCalPoints = 0;
            int numInputCalPaths = 0;
            int numOutputCalPaths = 0;

            // Clean up any old cal data
            if (inputCalFreqs != null) inputCalFreqs = null;
            if (inputCalValues != null) inputCalValues = null;
            if (outputCalFreqs != null) outputCalFreqs = null;
            if (outputCalValues != null) outputCalValues = null;

            // Open the file
            var rs = new FileStream(filePath + fileName, FileMode.Open);
            TextReader rd = new StreamReader(rs);

            // Read the data size values
            while (numInputCalPoints == 0 || numOutputCalPoints == 0 || numInputCalPaths == 0 || numOutputCalPaths == 0)
            {
                string fileString = rd.ReadLine();
                fileString = fileString.ToUpper();
                string[] columns = fileString.Split(',');
                if (columns[0].Contains("INPUT FREQS")) numInputCalPoints = Convert.ToInt32(columns[1]);
                if (columns[0].Contains("INPUT PATHS")) numInputCalPaths = Convert.ToInt32(columns[1]);
                if (columns[0].Contains("OUTPUT FREQS")) numOutputCalPoints = Convert.ToInt32(columns[1]);
                if (columns[0].Contains("OUTPUT PATHS")) numOutputCalPaths = Convert.ToInt32(columns[1]);
            }

            // Size the Frequency arrays
            inputCalFreqs = new double[numInputCalPoints];
            inputCalValues = new double[numInputCalPoints, numInputCalPaths];
            outputCalFreqs = new double[numOutputCalPoints];
            outputCalValues = new double[numOutputCalPoints, numOutputCalPaths];

            bool inputDataRead = false;
            bool outputDataRead = false;
            while (inputDataRead == false || outputDataRead == false)
            {
                string fileString = rd.ReadLine();
                fileString = fileString.ToUpper();
                if (fileString.Contains("INPUT DATA"))
                {
                    int linesRead = 0;
                    while (linesRead < numInputCalPoints)
                    {
                        fileString = rd.ReadLine();
                        fileString = fileString.ToUpper();
                        if (fileString.Contains("FREQUENCY") || fileString.Contains("//"))
                        {
                            // Skip Header or comment line
                        }
                        else
                        {
                            // Read the cal data
                            string[] columns = fileString.Split(',');
                            inputCalFreqs[linesRead] = Convert.ToDouble(columns[0]);
                            for (int j = 1; j < columns.Length && j < numInputCalPaths + 1; j++)
                            {
                                inputCalValues[linesRead, j-1] = Convert.ToDouble(columns[j]);
                            }
                            linesRead++;
                        }
                    }
                    inputDataRead = true;
                }
                if (fileString.Contains("OUTPUT DATA"))
                {
                    int linesRead = 0;
                    while (linesRead < numOutputCalPoints)
                    {
                        fileString = rd.ReadLine();
                        fileString = fileString.ToUpper();
                        if (fileString.Contains("FREQUENCY") || fileString.Contains("//"))
                        {
                            // Skip Header or comment line
                        }
                        else
                        {
                            // Read the cal data
                            string[] columns = fileString.Split(',');
                            outputCalFreqs[linesRead] = Convert.ToDouble(columns[0]);
                            for (int j = 1; j < columns.Length && j < numOutputCalPaths + 1; j++)
                            {
                                outputCalValues[linesRead, j-1] = Convert.ToDouble(columns[j]);
                            }
                            linesRead++;
                        }
                    }
                    outputDataRead = true;
                }
            }
            rd.Close();
            rs.Close();
        }
        private void useCalExample()
        {
            double measPout = -200;
            double targetPout = 0;
            double targetGain = 0;

            measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);

            Vsg.Modulation.BasebandPower = (targetPout - targetGain - Vsg.RF.Level + getInputAtten(Vsg.RF.Frequency, 0));

        }
        public void testCaldata()
        {
            // Tests the input and output path get cal data
            // run with breakpoint at end and check variables
            double[,] readCalData = new double[30, 3];
            double freq = 100e6;
            double freqStep = 300e6;
            int path = 1;
            for (int i = 0; i < 30; i++)
            {
                readCalData[i, 0] = (freq + i * freqStep)/1e9;
                readCalData[i, 1] = getInputAtten(freq + i * freqStep, path);
                readCalData[i, 2] = getOutputAtten(freq + i * freqStep, path);
            }
        }
        public void runPowerCal(string resourceDesc, string testWaveform, double[]freqs, double calPowerLevel, string filePath, 
            bool align = false, double nominalInLoss = 0, double nominalOutLoss = 0)
        {
            string s;
            string cmd;
            double[] referencePower = new double[freqs.Length];
            double[] inputLoss = new double[freqs.Length];
            double[] outputLoss = new double[freqs.Length];
            double[] error = new double[freqs.Length];

            for (int i = 0; i < freqs.Length; i++)
            {
                rfOffset[i] = 0;
                scaleOffset[i] = 1;
            }

            if (oRm == null)
                oRm = new ResourceManager();

            if (pm == null)
            {
                pm = new FormattedIO488();
                pm.IO = (IMessage)oRm.Open(resourceDesc, AccessMode.NO_LOCK, 2000, "");
                pm.IO.Timeout = 20000;
            }
            do
            {
                pm.WriteString("SYSTEM:ERROR?");
                s = pm.ReadString();
            } while (!s.ToUpper().Contains("NO ERROR"));
            pm.WriteString("SYST:PRES DEF");

            Vsg.RF.Frequency = freqs[0];
            Vsg.Modulation.BasebandFrequency = 0;
            Vsg.Modulation.BasebandPower = 0;
            Vsg.RF.Level = calPowerLevel;
            Vsg.Modulation.PlayArb(testWaveform, AgM938xStartEventEnum.AgM938xStartEventImmediate);

            DialogResult dialogResult = MessageBox.Show("Zero Power Sensor?", "Calibration", MessageBoxButtons.YesNo);
            if (dialogResult == DialogResult.Yes)
            {
                pm.WriteString("CAL:ZERO:TYPE INT");
                pm.WriteString("CAL");
                pm.WriteString("*OPC?");
                s = pm.ReadString();
                pm.WriteString("SYSTEM:ERROR?\n");
                s = pm.ReadString();
                if (!s.ToUpper().Contains("NO ERROR")) throw (new Exception("Power Meter Error:\n" + s));
            }

            pm.WriteString("*INIT:CONT OFF");
            pm.WriteString("INIT\n");
            pm.WriteString("FETC?\n");
            s = pm.ReadString();
            cmd = string.Format("CONF:POW:AC {0},DEF", calPowerLevel - nominalInLoss); // Power Meter Power level
            pm.WriteString(cmd);

            MessageBox.Show("Connect Power Sensor to end of DUT input cable", "Calibration");
            // Loop over Frequencies
            for (int i = 0; i < freqs.Length; i++)
            {
                cmd = string.Format("FREQ {0}", freqs[i]); // Power Meter Frequency
                pm.WriteString(cmd);
                // Set VSG to cal frequency
                Vsg.RF.Frequency = freqs[i];
                Vsg.Apply();
                Vsg.RF.WaitUntilSettled(100);
                if (align)
                {
                    // Perform power search
                    Vsg.Calibration.PowerSearch.DoPowerSearch(ref rfOffset[i], ref scaleOffset[i]);
                }
                Delay(100);
                // Read Power with power meter
                //pm.WriteString("INIT\n");
                //pm.WriteString("FETC?\n");
                //s = pm.ReadString();
                //error[i] = Convert.ToDouble(s);
                pm.WriteString("INIT\n");
                pm.WriteString("FETC?\n");
                s = pm.ReadString();
                referencePower[i] = Convert.ToDouble(s);
                //error[i] = error[i] - referencePower[i];
                inputLoss[i] = calPowerLevel - referencePower[i];
            }

            MessageBox.Show("Connect DUT input cable to DUT output cable.", "Calibration");

            // Get RMS value from playing waveform
            Vsg.Modulation.IQ.ArbInformation(testWaveform, ref arbSampleRate, ref arbRmsValue, arbScale, rfMarker, alcMarker);
            // Setup the VSA for a power measurement matching the ARB Settings
            Vsa.RF.Power = calPowerLevel + refLevelMargin - nominalInLoss - nominalOutLoss;
            Vsa.RF.Conversion = AgM9391ConversionEnum.AgM9391ConversionAuto;
            Vsa.RF.PeakToAverage = arbRmsValue;
            Vsa.RF.IFBandwidth = ifBandwidth;
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
            Vsa.PowerAcquisition.Bandwidth = sampleRate;
            Vsa.PowerAcquisition.Duration = 10e-3;  // Average power over complete frame
            Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            Vsa.PowerAcquisition.OffsetFrequency = 0;

            double measPout = -200;
            bool overload = false;

            for (int i = 0; i < freqs.Length; i++)
            {
                Vsg.RF.Frequency = freqs[i];
                if (align)
                    Vsg.Calibration.PowerSearch.UsePowerSearchResult(rfOffset[i], scaleOffset[i]);
                Vsg.Apply();
                Vsg.RF.WaitUntilSettled(100);
                Delay(100);

                Vsa.RF.Frequency = freqs[i];
                Vsa.Apply();
                Vsa.WaitUntilSettled(100);

                if (align)
                    Vsa.Calibration.Amplitude.Align(AgM9391AmplitudeAlignmentTypeEnum.AgM9391AmplitudeAlignmentTypeCurrentState);

                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at runPowerCal"));
                outputLoss[i] = referencePower[i] - measPout;
            }

            Vsg.Modulation.Stop();
            Vsg.Modulation.PlayArb(testWaveform, AgM938xStartEventEnum.AgM938xStartEventImmediate);
            Vsa.RF.Power = calPowerLevel + refLevelMargin - nominalOutLoss;

            // Verify Calibratoin by setting 0 dBm and measuring error
            for (int i = 0; i < freqs.Length; i++)
            {
                Vsg.RF.Frequency = freqs[i];
                Vsg.RF.Level = calPowerLevel + inputLoss[i];
                if (align)
                    Vsg.Calibration.PowerSearch.DoPowerSearch(ref rfOffset[i], ref scaleOffset[i]);
                Vsg.Apply();
                Vsg.RF.WaitUntilSettled(100);
                Delay(100);

                Vsa.RF.Frequency = freqs[i];
                Vsa.Apply();
                Vsa.WaitUntilSettled(100);
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at runPowerCal"));
                error[i] = measPout + outputLoss[i];
            }

            // Create the cal data file
            TextWriter calFile = new StreamWriter(filePath, false);
            calFile.WriteLine(string.Format("Input Freqs, {0}", freqs.Length));
            calFile.WriteLine(string.Format("Input Paths, {0}", 1));
            calFile.WriteLine(string.Format("Output Freqs, {0}", freqs.Length));
            calFile.WriteLine(string.Format("Output Paths, {0}", 1));

            calFile.WriteLine("");
            calFile.WriteLine("Input Data");
            calFile.WriteLine("Frequency, Path 1");
            for (int i = 0; i < freqs.Length; i++)
            {
                calFile.WriteLine(string.Format("{0},{1}", freqs[i], inputLoss[i]));
            }

            calFile.WriteLine("");
            calFile.WriteLine("Output Data");
            calFile.WriteLine("Frequency, Path 1");
            for (int i = 0; i < freqs.Length; i++)
            {
                calFile.WriteLine(string.Format("{0},{1}", freqs[i], outputLoss[i]));
            }

            calFile.WriteLine("");
            calFile.WriteLine("Error Data");
            calFile.WriteLine("Frequency, Path 1");
            for (int i = 0; i < freqs.Length; i++)
            {
                calFile.WriteLine(string.Format("{0},{1}", freqs[i], error[i]));
            }
            calFile.Close();
            //System.Diagnostics.Process.Start(filePath);
        }
    }
}
